<?php

/**
 * Seymour - A feed reading library for PHP
 * 
 * <b>Feed_mapping.php</b>
 * 
 * Defines several classes used to map Atom and RSS feeds to an object model.
 * The main classes are based on Atom 1.0, all other supported formats are then
 * mapped to the Atom objects.
 * 
 * Currently supports Atom 1.0, 0.3, RSS 2.0, 1.0, 0.9x
 * 
 * Since this is an alpha version there are probably lots of inconsistencies in
 * the way elements are treated. Any suggestions/comments/bugs are welcome on
 * the Sourceforge forums.
 * 
 * <b>Adding support for a format</b>
 * 
 * All the objects in this file are created according to the requirements of
 * XML_Unserializer. What the unserializer does is parse the XML of the feed,
 * and for each element, it searches for an object with the same name, it then
 * searches the object for variables matching the names of the child elements,
 * or for functions named set[childelement], and sets the variable or runs the
 * function accordingly. 
 * 
 * For example if a raw feed looked like this:
 * <code> 
 * <parent>
 * <child1>Some CDATA</child1>
 * <child2>More CDATA</child2>
 * </parent>
 * </code>
 * 
 * And your object definitions looked like this:
 * <code>
 * class parent {
 *   var $child1;
 *   var $child2processed;
 *   function setChild2($child2) {
 *     $this->child2processed = $child2 . ', and my code added this';
 *   }
 * }
 * </code>
 * 
 * Then after unserializer runs, you will have the following instances available
 * <code>
 * echo($parent->child1); 
 * echo($parent->child2processed); 
 * </code>
 * 
 * Which prints:
 * 
 * Some CDATA
 * More CDATA, and my code added this
 *
 * <b>LICENSE:</b>
 * 
 * This file is part of Seymour.
 * 
 * Seymour is free software; you can redistribute it and/or modify it under the
 * terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 * 
 * Seymour is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 * A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with Seymour; if not, write to the Free Software Foundation, Inc., 51
 * Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 * 
 * Included packages such as those from the PEAR library and Simpletest are
 * copyright their respective authors and are subject to their own licenses as
 * specified in their source files.
 *
 * @package     Seymour
 * @subpackage	FormatMapping
 * @author      Tom Walter <evilpuppet@users.sourceforge.net>
 * @copyright   Copyright (C) 2005 Tom Walter
 * @license     http://www.gnu.org/licenses/lgpl.txt  GNU Lesser General Public License
 * @version     CVS: $Id: feed_mapping.php,v 1.1 2005/11/06 12:26:29 evilpuppet Exp $
 * @link        http://seymour.sourceforge.net/
 */


/**
 * Regular expression for matching dates in ISO 8601 format
 * 
 * @link http://www.w3.org/TR/NOTE-datetime ISO 8601 spec
 */
define('DATE_ISO_REGEX', '/^(\d{4})-(\d{2})-(\d{2})T(\d{2}):(\d{2}):(\d{2})(\.\d+)?(Z|(((\+|-)\d{2}):(\d{2})))$/');

/**
 * PHP date() formatting string for ISO 8601
 * 
 * @link http://www.w3.org/TR/NOTE-datetime ISO 8601 spec
 */
define('DATE_ISO_FORMAT', 'Y-m-d\TH:i:s\Z');

/**
 * PHP date() formatting string for RFC 822
 * 
 * @link http://www.w3.org/Protocols/rfc2616/rfc2616-sec3.html#sec3.3.1 RFC 822
 * spec
 */
define('DATE_RFC_FORMAT', 'D, d M Y H:i:s \U\T\C');



// Begin Atom 1.0 objects ----------------------------------------------------

/**
 * atomCommonAttributes (Atom 1.0)
 * 
 * @link http://www.atompub.org/2005/08/17/draft-ietf-atompub-format-11.html#rfc.section.2 Atom 1.0 spec
 */
class CommonAttributes {
	var $xml_base;
	var $xml_lang;
	var $_content;
	
	// getters ------
	function &get() {
		return $this->_content;
	}
	
	function &getXMLBase() {
		return $this->xml_base;
	}
	
	function &getXMLLang() {
		return $this->xml_lang;
	}
}


/**
 * atomTextConstruct (Atom 1.0)
 * 
 * @link http://www.atompub.org/2005/08/17/draft-ietf-atompub-format-11.html#rfc.section.3.1 Atom 1.0 spec
 */
class TextConstruct extends CommonAttributes {
	var $type = 'text'; // assume plain text
	
	//getters -------
	function &getType() {
		return $this->type;
	}
	
	function getPlainText() {
		$tmp = removeBadTagWithContent($this->_content);
		return strip_tags($tmp);
	}
	
	function getSafeHTML() {
		return removeBadTags($this->_content);
	}
	
	/**
	 * Helper function that returns all links in the text.
	 * 
	 * @todo To be completed
	 */
	function findLinks() {
	}
}

/**
 * atomPlainTextConstruct (Atom 1.0)
 * 
 * @link http://www.atompub.org/2005/08/17/draft-ietf-atompub-format-11.html#rfc.section.3.1 Atom 1.0 spec
 */
class PlainTextConstruct extends TextConstruct {
}

/**
 * atomXHTMLTextConstruct (Atom 1.0)
 * 
 * @link http://www.atompub.org/2005/08/17/draft-ietf-atompub-format-11.html#rfc.section.3.1 Atom 1.0 spec
 */
class XHTMLTextConstruct extends TextConstruct {
}


/**
 * atomPersonConstruct (Atom 1.0, 0.3)
 * 
 * @link http://www.atompub.org/2005/08/17/draft-ietf-atompub-format-11.html#rfc.section.3.2 Atom 1.0 spec
 * @link http://www.mnot.net/drafts/draft-nottingham-atom-format-02.html#rfc.section.3.2 Atom 0.3 spec
 */
class PersonConstruct {
	var $name;
	var $uri;
	var $email;
	
	/**
	 * Maps the Atom 0.3 'url' to Atom 1.0 'uri'
	 * 
	 * @param	string	$url	Atom 0.3 url
	 * @return	void
	 * @ignore
	 */
	function setUrl($url) {
		$this->uri = $url;
	}
	
	// getters -------------
	function &get() {
		return $this->name;
	}
	
	function &getName() {
		return $this->name;
	}
	
	function &getUri() {
		return $this->uri;
	}
	
	function &getEmail() {
		return $this->email;
	}
}

//class Name {
//}
//
//class Uri {
//}
//
//class Email {
//}


/**
 * atomDateConstruct (Atom 1.0, 0.3)
 * 
 * Regardless of the feed format and timezone, the date is always converted to a
 * UTC date. You can then use the get functions to return the date in the format
 * you require and convert to the timezone you require.
 * 
 * @link http://www.atompub.org/2005/08/17/draft-ietf-atompub-format-11.html#rfc.section.3.3 Atom 1.0 spec
 * @link http://www.mnot.net/drafts/draft-nottingham-atom-format-02.html#rfc.section.3.3 Atom 0.3 spec
 */
class DateConstruct {
	var $timestamp;
	
	/**
	 * Converts the contents of a date in plain text into a timestamp in UTC.
	 * 
	 * @param	string	$_content	Contents of a date element in plain text.
	 * @return	void
	 * @ignore
	 */
	function set_Content($_content) {
		putenv ('TZ=CUT0GDT'); // we store all dates in UTC
		
		$matches = array();
		if (preg_match(DATE_ISO_REGEX, $_content, $matches)) {
			$hours = $matches[4];
			if (isset($matches[10])) $hours -= $matches[10]; // subtract offset to convert date to UTC
			$mins = $matches[5];
			if (isset($matches[12])) $mins -= $matches[12]; // subtract offset to convert date to UTC
			$this->timestamp = mktime($hours, $mins, $matches[6], $matches[2], $matches[3], $matches[1]);
		} else {
			$this->timestamp = strtotime($_content); // attempt to parse date into timestamp
		}
	}
	
	// getters ------
	
	function &get() {
		return $this->getTimestamp();
	}
	
	/**
	 * Returns date in ISO 8601 format. Format used by Atom.
	 * 
	 * @return	string	Date as ISO 8601
	 * 
	 * @link http://www.w3.org/TR/NOTE-datetime
	 */
	function &getAtom() {
		$date = date(DATE_ISO_FORMAT, $this->timestamp);
		return $date;
	}
	
	/**
	 * Returns date in RFC_822. Format used by RSS.
	 * 
	 * @return	string	Date as RFC 822
	 * 
	 * @link http://www.w3.org/Protocols/rfc2616/rfc2616-sec3.html#sec3.3.1
	 */
	function &getRSS() {
		$date = date(DATE_RFC_FORMAT, $this->timestamp);
		return $date;
	}
	
	/**
	 * Returns date as Unix timestamp
	 * 
	 * @return	int		Timestamp
	 */
	function &getTimestamp() {
		return $this->timestamp;
	}
}


/**
 * atom:feed (Atom 1.0, 0.3)
 * 
 * The feed object is the parent of all the other elements. The RSS <channel>
 * and <rss> elements are also mapped to this.
 * 
 * Some elements of the various RSS formats and Atom 0.3 that don't have any
 * Atom 1.0 equivalents are also included in this object, and are named as per
 * their corresponding formats.
 * 
 * @link http://www.atompub.org/2005/08/17/draft-ietf-atompub-format-11.html#rfc.section.4.1.1 Atom 1.0 spec
 * @link http://www.mnot.net/drafts/draft-nottingham-atom-format-02.html#rfc.section.4 Atom 0.3 spec
 * @link http://www.feedvalidator.org/docs/rss2.html#whatIsRss RSS 2.0 channel
 * spec
 */
class Feed extends CommonAttributes {
	
	/**#@+
	 * Atom 1.0 properties
	 */
	var $xmlns;
	var $authors;
	var $categories;
	var $contributors;
	var $generator;
	var $icon;
	var $id;
	var $links;
	var $logo;
	var $rights;
	var $subtitle;
	var $title;
	var $updated;
	var $entries;
	/**#@-*/ // end Atom 1.0 props
	
	/**#@+
	 * Atom 0.3 exclusive properties
	 */
	var $info;
	/**#@-*/ // end Atom 0.3 props
	
	/**#@+
	 * RSS 2.0 exclusive properties
	 */
	var $version;
	var $language;
	var $docs;
	var $published;
	var $cloud;
	var $ttl;
	var $textInput;
	var $skipHours;
	var $skipDays;
	/**#@-*/ // end RSS 2.0 props
	
	/**#@+
	 * Rename elements to make it obvious they are arrays, not single objects.
	 * 
	 * @ignore
	 */
	function setAuthor($author) {
		$this->authors = $author;
	}
	
	function setLink($link) {
		$this->links = $link;
	}
	
	function setCategory($category) {
		$this->categories = $category;
	}
	
	function setContributor($contributor) {
		$this->contributors = $contributor;
	}
	
	function setEntry($entry) {
		$this->entries = $entry;
	}
	/**#@-*/ // end renaming
	
	/**#@+
	 * Map Atom 0.3 element to 1.0 equivalent
	 * 
	 * @ignore
	 */
	 
	function setTagline($tagline) {
		$this->subtitle = $tagline;
	}
	
	function setCopyright($copyright) {
		$this->rights = $copyright;
	}
	
	function setModified($modified) {
		$this->updated = $modified;
	}
	
	/**#@-*/ // end Atom 0.3 mapping
	
	
	// getters ------
	
	function &getXmlns() {
		return $this->xmlns;
	}
	
	function &getAuthors() {
		return $this->authors;
	}
	
	function &getMainAuthor() {
		if (isset($this->authors[0])) {
			return $this->authors[0];
		} else {
			return null;
		}
	}
	
	function &getCategories() {
		return $this->categories;
	}
	
	function &getContributors() {
		return $this->contributors;
	}
	
	function &getGenerator() {
		return $this->generator;
	}
	
	function &getIcon() {
		return $this->icon;
	}
	
	function &getId() {
		if (isset($this->id)) {
			return $this->id->get();
		} else {
			return null;
		}
	}
	
	function &getLinks() {
		return $this->links;
	}
	
	function &getSelfLink() {
		return findSelfHref($this->links);
	}
	
	function &getAltLink() {
		return findAltHref($this->links);
	}
	
	function &getLogo() {
		return $this->logo;
	}
				
	function &getRights() {
		if (isset($this->rights)) {
			return $this->rights->get();
		} else {
			return null;
		}
	}
	
	function &getSubtitle() {
		if (isset($this->subtitle)) {
			return $this->subtitle->get();
		} else {
			return null;
		}
	}
	
	function &getTitle() {
		if (isset($this->title)) {
			return $this->title->get();
		} else {
			return null;
		}
	}
	
	function &getUpdated() {
		if (isset($this->updated)) {
			return $this->updated->get();
		} else {
			return null;
		}
	}
	
	function &getEntries() {
		return $this->entries;
	}
	
	function &getInfo() {
		if (isset($this->info)) {
			return $this->info->get();
		} else {
			return null;
		}
	}
	
	function &getVersion() {
		return $this->version;
	}
	
	function &getLanguage() {
		return $this->language;
	}
	
	function &getDocs() {
		if (isset($this->docs)) {
			return $this->docs->get();
		} else {
			return null;
		}
	}
	
	function &getPublished() {
		if (isset($this->published)) {
			return $this->published->get();
		} else {
			return null;
		}
	}
	
	function &getCloud() {
		return $this->cloud;
	}
	
	function &getTtl() {
		if (isset($this->ttl)) {
			return $this->ttl->get();
		} else {
			return null;
		}
	}
	
	function &getTextInput() {
		return $this->textInput;
	}
	
	function &getSkipHours() {
		if (isset($this->skipHours)) {
			return $this->skipHours->get();
		} else {
			return null;
		}
	}
	
	function &getSkipDays() {
		if (isset($this->skipDays)) {
			return $this->skipDays->get();
		} else {
			return null;
		}
	}
	
	/**
	 * Performs some processing on the feed object after all the children have
	 * been created.
	 * 
	 * @access	private
	 */
	function __wakeup() {
		if (!isset($this->updated)) {
			$this->updated = $this->published;
		}
		
		if (!isset($this->published)) {
			$this->published = $this->updated;
		}
	}
	
}

/**
 * atom:entry (Atom 1.0, 0.3)
 * 
 * A feed entry. The RSS <item> element is mapped to this.
 * 
 * @link http://www.atompub.org/2005/08/17/draft-ietf-atompub-format-11.html#rfc.section.4.1.2 Atom 1.0 spec
 * @link http://www.mnot.net/drafts/draft-nottingham-atom-format-02.html#rfc.section.4.13 Atom 0.3 spec
 * @link http://www.feedvalidator.org/docs/rss2.html#hrelementsOfLtitemgt RSS
 * 2.0 item spec
 */
class Entry extends CommonAttributes {
	
	/**#@+
	 * Atom 1.0 properties
	 */
	var $authors;
	var $categories;
	var $content;
	var $contributors;
	var $id;
	var $links;
	var $published;
	var $rights;
	var $source;
	var $summary;
	var $title;
	var $updated;
	/**#@-*/ // end Atom 1.0 props
	
	/**#@+
	 * RSS 2.0 exclusive properties
	 */
	var $comments;
	/**#@-*/ // end RSS 2.0 props
	
	/**#@+
	 * Rename elements to make it obvious they are arrays, not single objects.
	 * 
	 * @ignore
	 */
	function setAuthor($author) {
		$this->authors = $author;
	}
	
	function setLink($link) {
		$this->links = $link;
	}
	
	function setCategory($category) {
		$this->categories = $category;
	}
	
	function setContributor($contributor) {
		$this->contributors = $contributor;
	}
	/**#@-*/ // end renaming
	
	/**#@+
	 * Map Atom 0.3 element to 1.0 equivalent
	 * 
	 * @ignore
	 */
	 
	function setModified($modified) {
		$this->updated = $modified;
	} 
	
	function setIssued($issued) {
		$this->published = $issued;
	}
	
	function setCreated($created) {
		$this->published = $created;
	}
	
	/**#@-*/ // end Atom 0.3 mapping
	
	
	// getters -------
	
	function &getAuthors() {
		return $this->authors;
	}
	
	function &getMainAuthor() {
		if (isset($this->authors[0])) {
			return $this->authors[0];
		} else {
			return null;
		}
	}
	
	function &getCategories() {
		return $this->categories;
	}
	
	function &getContent() {
		if (isset($this->content)) {
			return $this->content->getSafeHTML();
		} else {
			return null;
		}
	}
	
	function &getContributors() {
		return $this->contributors;
	}
	
	function &getId() {
		if (isset($this->id)) {
			return $this->id->get();
		} else {
			return null;
		}
	}
	
	function &getLinks() {
		return $this->links;
	}
	
	function &getAltLink() {
		return findAltHref($this->links);
	}
	
	function &getPublished() {
		if (isset($this->published)) {
			return $this->published->get();
		} else {
			return null;
		}
	}
	
	function &getRights() {
		if (isset($this->rights)) {
			return $this->rights->get();
		} else {
			return null;
		}
	}
	
	function &getSource() {
		return $this->source;
	}
	
	function &getSummary() {
		if (isset($this->summary)) {
			return $this->summary->getSafeHTML();
		} else {
			return null;
		}
	}
	
	function &getTitle() {
		if (isset($this->title)) {
			return $this->title->getPlainText();
		} else {
			return null;
		}
	}
	
	function &getUpdated() {
		if (isset($this->updated)) {
			return $this->updated->get();
		} else {
			return null;
		}
	}
	
	function &getComments() {
		if (isset($this->comments)) {
			return $this->comments->get();
		} else {
			return null;
		}
	}
	
	/**
	 * Performs some processing on the entry object after all the children have
	 * been created.
	 * 
	 * @access	private
	 */
	function __wakeup() {
		if (!isset($this->updated)) {
			$this->updated = $this->published;
		}
		
		if (!isset($this->published)) {
			$this->published = $this->updated;
		}
		
		// if a short description doesn't exist, make one from the body text
		if (!isset($this->summary)) {
			$this->summary = new Summary();
			$this->summary->_content = $this->content->getPlainText();
		}		
		$this->summary->_content = summariseText($this->summary->_content);
	}
}


/**#@+
 * atom:content (Atom 1.0)
 * 
 * @link http://www.atompub.org/2005/08/17/draft-ietf-atompub-format-11.html#atomContent Atom 1.0 spec
 */

class InlineTextContent extends TextConstruct {
}

class InlineXHTMLContent extends TextConstruct {
}

class InlineOtherContent extends TextConstruct {
}

class OutOfLineContent extends TextConstruct {
	var $src;
	
	//getters -------
	function &getSrc() {
		return $this->src;
	}
}

class Content extends OutOfLineContent {	
}

/**#@-*/ // end atom:content


/**
 * atom:author (Atom 1.0, 0.3)
 * 
 * Also applies to RSS item <author>
 * 
 * @link http://www.atompub.org/2005/08/17/draft-ietf-atompub-format-11.html#rfc.section.4.2.1 Atom 1.0 spec
 * @link http://www.mnot.net/drafts/draft-nottingham-atom-format-02.html#rfc.section.4.5 Atom 0.3 spec
 * @link http://www.feedvalidator.org/docs/rss2.html#ltauthorgtSubelementOfLtitemgt RSS 2.0 spec
 */
class Author extends PersonConstruct { 
	
	/**
	 * Map RSS 2.0 <author> to Atom 1.0
	 * 
	 * @ignore
	 */
	function set_Content($_content) {
		$this->email = $_content;
	}
}

/**
 * atom:category (Atom 1.0)
 * 
 * Also applies to RSS <category>
 * 
 * @link http://www.atompub.org/2005/08/17/draft-ietf-atompub-format-11.html#rfc.section.4.2.2 Atom 1.0 spec
 * @link http://www.feedvalidator.org/docs/rss2.html#syndic8 RSS 2.0 spec
 */
class Category {
	var $term;
	var $scheme;
	var $label;
	
	/**#@+
	 * Map RSS 2.0 <category> to Atom 1.0
	 * 
	 * @ignore
	 */
	function setDomain($domain) { 
		$this->scheme = $domain;
	}

	function set_Content($_content) {
		$this->term = $_content;
	}
	/**#@-*/ // end RSS 2.0 mapping
	
	
	// getters ------
	
	function &get() {
		return $this->term;
	}
	
	function &getTerm() {
		return $this->term;
	}
	
	function &getScheme() {
		return $this->scheme;
	}
	
	function &getLabel() {
		return $this->label;
	}

}

/**
 * atom:contributor (Atom 1.0, 0.3)
 * 
 * @link http://www.atompub.org/2005/08/17/draft-ietf-atompub-format-11.html#rfc.section.4.2.3 Atom 1.0 spec
 * @link http://www.mnot.net/drafts/draft-nottingham-atom-format-02.html#rfc.section.4.6 Atom 0.3 spec
 */
class Contributor extends PersonConstruct {
}

/**
 * atom:generator (Atom 1.0, 0.3)
 * 
 * Also applies to RSS <generator>
 * 
 * @link http://www.atompub.org/2005/08/17/draft-ietf-atompub-format-11.html#rfc.section.4.2.4 Atom 1.0 spec
 * @link http://www.mnot.net/drafts/draft-nottingham-atom-format-02.html#rfc.section.4.9 Atom 0.3 spec
 * @link http://www.feedvalidator.org/docs/rss2.html#optionalChannelElements RSS
 * 2.0 generator spec
 */
class Generator {
	var $name;
	var $uri;
	var $version;
	
	/**
	 * Rename for clarity
	 * 
	 * @ignore
	 */
	function set_Content($_content) {
		$this->name = $_content;
	}
	
	/**
	 * Map Atom 0.3 to 1.0
	 * 
	 * @ignore
	 */
	function setUrl($url) {
		$this->uri = $url;
	}
	
	// getters --------
	function &get() {
		return $this->name;
	}
	
	function &getName() {
		return $this->name;
	}
	
	function &getUri() {
		return $this->uri;
	}
	
	function &getVersion() {
		return $this->version;
	}
}

/**
 * A parent class that maps Atom image elements to the RSS image format.
 * 
 * This is backwards to every other mapping. But necessary in this one case
 * since the RSS image definition is more complicated than Atom's
 */
class atomImage extends Image {
	
	/**
	 * Map the Atom image url to the RSS image url
	 * 
	 * @ignore
	 */
	function set_Content($_content) {
		$this->url = $_content;
	}
}

/**
 * atom:icon (Atom 1.0)
 *  
 * @link http://www.atompub.org/2005/08/17/draft-ietf-atompub-format-11.html#rfc.section.4.2.5 Atom 1.0 spec
 */
class Icon extends atomImage {
	
	/**
	 * Defaults the attributes of an atom image to conform to RSS image format
	 * 
	 * @ignore
	 */
	function Icon() {
		$this->width = 32;
		$this->height = 32;
	}
}

/**
 * atom:id (Atom 1.0, 0.3)
 * 
 * @link http://www.atompub.org/2005/08/17/draft-ietf-atompub-format-11.html#rfc.section.4.2.6 Atom 1.0 spec
 * @link http://www.mnot.net/drafts/draft-nottingham-atom-format-02.html#rfc.section.4.8 Atom 0.3 spec
 */
class Id {
	var $id;
	
	/**
	 * Rename for clarity
	 * 
	 * @ignore
	 */
	function set_Content($_content) {
		$this->id = $_content;
	}
	
	function &get() {
		return $this->id;
	}
}

/**
 * atom:link (Atom 1.0, 0.3)
 * 
 * Also applies to RSS item <link>
 * 
 * @link http://www.atompub.org/2005/08/17/draft-ietf-atompub-format-11.html#rfc.section.4.2.7 Atom 1.0 spec
 * @link http://www.mnot.net/drafts/draft-nottingham-atom-format-02.html#rfc.section.4.4 Atom 0.3 spec
 * @link http://www.feedvalidator.org/docs/rss2.html#requiredChannelElements RSS
 * 2.0 link spec
 */
class Seymour_Link {
	var $href;
	var $rel;
	var $type;
	var $hreflang;
	var $title;
	var $length;
	
	/**
	 * Map an RSS 2.0 link to Atom
	 * 
	 * @ignore
	 */
	function set_Content($_content) {
		$this->href = $_content;
		$this->rel = 'alternate';
		$this->type = 'text\html';
	}
	
	//getters -------
	function &get() {
		return $this->href;
	}
	
	function &getHref() {
		return $this->href;
	}
	
	function &getRel() {
		return $this->rel;
	}
	
	function &getType() {
		return $this->type;
	}
	
	function &getHrefLang() {
		return $this->hreflang;
	}
	
	function &getTitle() {
		return $this->title;
	}
	
	function &getLength() {
		return $this->length;
	}
}

/**
 * atom:logo (Atom 1.0)
 * 
 * @link http://www.atompub.org/2005/08/17/draft-ietf-atompub-format-11.html#rfc.section.4.2.8 Atom 1.0 spec
 */
class Logo extends atomImage {
	
	/**
	 * Defaults the attributes of an atom image to conform to RSS image format
	 * 
	 * @ignore
	 */
	function Logo() {
		$this->width = 32;
		$this->height = 64;
	}
}

/**
 * atom:published (Atom 1.0)
 * 
 * @link http://www.atompub.org/2005/08/17/draft-ietf-atompub-format-11.html#rfc.section.4.2.9 Atom 1.0 spec
 */
class Published extends DateConstruct {
}

/**
 * atom:rights (Atom 1.0)
 * 
 * @link http://www.atompub.org/2005/08/17/draft-ietf-atompub-format-11.html#rfc.section.4.2.10 Atom 1.0 spec
 */
class Rights extends TextConstruct { 
}

/**
 * atom:source (Atom 1.0)
 * 
 * Since this contains almost exactly the same elements as an atom:feed, we
 * simply extend that object.
 * 
 * Also applies to RSS item <source>
 * 
 * @link http://www.atompub.org/2005/08/17/draft-ietf-atompub-format-11.html#rfc.section.4.2.11 Atom 1.0 spec
 * @link http://www.feedvalidator.org/docs/rss2.html#ltsourcegtSubelementOfLtitemgt RSS 2.0 spec
 */
class Source extends Feed {
	
	/**#@+
	 * Map RSS 2.0 'source' properties to Atom 1.0
	 * 
	 * @ignore
	 */
	function setUrl($url) {
		$selfLink = new Seimour_Link();
		$selfLink->href = $url;
		$selfLink->rel = 'self';
		$selfLink->type = 'application/xml';
		$this->links[] = $selfLink;
	}
	
	function set_Content($_content) {
		$this->title = $_content;
	}
	/**#@-*/ // end RSS 2.0 mapping
}

/**
 * atom:subtitle (Atom 1.0)
 * 
 * @link http://www.atompub.org/2005/08/17/draft-ietf-atompub-format-11.html#rfc.section.4.2.12 Atom 1.0 spec
 */
class Subtitle extends TextConstruct {
}

/**
 * atom:summary (Atom 1.0, 0.3)
 * 
 * @link http://www.atompub.org/2005/08/17/draft-ietf-atompub-format-11.html#rfc.section.4.2.13 Atom 1.0 spec
 * @link http://www.mnot.net/drafts/draft-nottingham-atom-format-02.html#rfc.section.4.13.9 Atom 0.3 spec
 */
class Summary extends TextConstruct {
}

/**
 * atom:title (Atom 1.0, 0.3)
 * 
 * @link http://www.atompub.org/2005/08/17/draft-ietf-atompub-format-11.html#rfc.section.4.2.14 Atom 1.0 spec
 * @link http://www.mnot.net/drafts/draft-nottingham-atom-format-02.html#rfc.section.4.13.1 Atom 0.3 spec
 */
class Title extends TextConstruct {
}

/**
 * atom:updated (Atom 1.0)
 * 
 * @link http://www.atompub.org/2005/08/17/draft-ietf-atompub-format-11.html#rfc.section.4.2.15 Atom 1.0 spec
 */
class Updated extends DateConstruct {
}

// End Atom 1.0 objects ----------------------------------------------------



// Begin Atom 0.3 exclusive objects ----------------------------------------

/**
 * atom:tagline (Atom 0.3)
 * 
 * Renamed to atom:subtitle in 1.0
 * 
 * @link http://www.mnot.net/drafts/draft-nottingham-atom-format-02.html#rfc.section.4.7 Atom 0.3 spec
 */
class Tagline extends Subtitle {
}

/**
 * atom:copyright (Atom 0.3)
 * 
 * Renamed to atom:rights in 1.0
 * 
 * Also applies to RSS <copyright> element
 * 
 * @link http://www.mnot.net/drafts/draft-nottingham-atom-format-02.html#rfc.section.4.10 Atom 0.3 spec
 * @link http://www.feedvalidator.org/docs/rss2.html#optionalChannelElements RSS
 * 2.0 spec
 */
class Copyright extends Rights {
}

/**
 * atom:info (Atom 0.3)
 * 
 * Removed in 1.0
 * 
 * @link http://www.mnot.net/drafts/draft-nottingham-atom-format-02.html#rfc.section.4.11 Atom 0.3 spec
 */
class Info extends TextConstruct {
}

/**
 * atom:modified (Atom 0.3)
 * 
 * Renamed to atom:updated in 1.0 
 * 
 * @link http://www.mnot.net/drafts/draft-nottingham-atom-format-02.html#rfc.section.4.12 Atom 0.3 spec
 */
class Modified extends Updated {
}

/**
 * atom:issued (Atom 0.3)
 * 
 * Renamed to atom:published in 1.0
 * 
 * @link http://www.mnot.net/drafts/draft-nottingham-atom-format-02.html#rfc.section.4.13.7 Atom 0.3 spec
 */
class Issued extends Published {
}

/**
 * atom:created (Atom 0.3)
 * 
 * Removed in 1.0, but seems similar to atom:published, so we map it to
 * that.
 * 
 * @link http://www.mnot.net/drafts/draft-nottingham-atom-format-02.html#rfc.section.4.13.8 Atom 0.3 spec
 */
class Created extends Published {
}

// End Atom 0.3 exclusive objects ----------------------------------------



// Begin RSS 2.0 exclusive objects ----------------------------------------

/**
 * <rss> (RSS 2.0, 0.91, 0.92)
 * 
 * Equivalent to atom:feed
 * 
 * @link http://www.feedvalidator.org/docs/rss2.html#whatIsRss RSS 2.0 spec
 */
class Rss extends Feed {
	
	/**
	 * In order to mimic Atom, we have to copy the children of the channel
	 * element up into the main feed element, since atom has no equivalent of
	 * channels.
	 * 
	 * This function is called after the subordinate channel object has already
	 * been created.
	 * 
	 * @ignore
	 */
	function setChannel($channel) {
		foreach ($channel as $prop => $propVal) {
			$setMethod = 'set'.$prop;
			if (method_exists($this, $setMethod)) {
				call_user_func(array(&$this, $setMethod), $propVal);
			} else {
				$this->$prop = $propVal;
			}
		}
	}
	
	/**
	 * Needed to ensure that an object is created
	 * 
	 * @ignore
	 */
	function setTextinput($textInput) {
		$this->textInput = $textInput;
	}
	
}

/**
 * <channel> (RSS 2.0, 0.91, 0.92)
 * 
 * Also equivalent to atom:feed
 * 
 * @link http://www.feedvalidator.org/docs/rss2.html#whatIsRss RSS 2.0 spec
 */
class Channel extends Feed {
	
	/**#@+
	 * Map RSS 2.0 elements to atom:feed equivalents
	 * 
	 * @ignore
	 */
	 
	function setDescription($description) {
		$this->subtitle = $description;
	}
	
	function setPubDate($pubDate) {
		$this->published = $pubDate;
	}
	
	function setLastBuildDate($lastBuildDate) {
		$this->updated = $lastBuildDate;
	}
	
	function setManagingEditor($managingEditor) {
		$this->authors = array_merge($this->authors, $managingEditor);
	}
	
	function setWebMaster($webMaster) {
		$this->authors = array_merge($this->authors, $webMaster);
	}
	
	function setImage($image) {
		$this->logo = $image;
	}
	
	function setItem($item) {
		$this->entries = $item;
	}
	
	/**#@-*/ // end RSS 2.0 mapping
	
	/**#@+
	 * Map dublin core elements to atom:feed equivalents
	 * 
	 * @ignore
	 */
	
	function setDc_title($dc_item) {
		$this->title = $dc_item;
	}
	
	function setDc_creator($dc_item) {
		$this->authors = $dc_item;
	}
	
	function setDc_subject($dc_item) {
		$this->categories = $dc_item;
	}
	
	function setDc_description($dc_item) {
		$this->summary = $dc_item;
	}
	
	function setDc_contributor($dc_item) {
		$this->contributors = $dc_item;
	}
	
	function setDc_date($dc_item) {
		$this->published = $dc_item;
	}
	
	function setDc_identifier($dc_item) {
		$this->id = $dc_item;
	}
	
	function setDc_source($dc_item) {
		$this->source = $dc_item;
	}
	
	function setDc_language($dc_item) {
		$this->language = $dc_item;
	}
	
	function setDc_rights($dc_item) {
		$this->rights = $dc_item;
	}
	
	/**#@-*/ // end dublin core mapping
	
}

/**#@+
 * RSS 2.0 required element
 * 
 * @link http://www.feedvalidator.org/docs/rss2.html#requiredChannelElements RSS
 * 2.0 required elements spec
 */
 
/**
 * <description> (RSS)
 */
class Description extends Summary {
}

/**#@-*/ // end RSS 2.0 required

/**
 * A parent class that maps RSS-style people to their Atom equivalent.
 */
class RSSPerson extends PersonConstruct {
	
	/**
	 * The content of an RSS person element is always an email address
	 * 
	 * @ignore
	 */
	function set_Content($_content) {
		$this->email = $_content;
	}
	
}

/**#@+
 * RSS 2.0 optional element
 * 
 * @link http://www.feedvalidator.org/docs/rss2.html#optionalChannelElements
 */

//class Language {
//}

/** 
 * <managingEditor> (RSS)
 */
class ManagingEditor extends RSSPerson {
}

/**
 * <webMaster> (RSS)
 */
class WebMaster extends RSSPerson {
}

/**
 * <pubDate> (RSS)
 */
class PubDate extends Published {
}

/**
 * <lastBuildDate> (RSS)
 */
class LastBuildDate extends Updated {
}

/**
 * <docs> (RSS)
 */
class Docs {
	var $url;
	
	/**
	 * Content of a <docs> element is always a URL
	 * 
	 * @ignore
	 */
	function set_Content($_content) {
		$this->url = $_content;
	}
	
	//getters -------
	function &get() {
		return $this->url;
	}
	
	function &getUrl() {
		return $this->url;
	}
}

/**
 * <cloud> (RSS)
 * @link http://www.feedvalidator.org/docs/rss2.html#ltcloudgtSubelementOfLtchannelgt RSS 2.0 cloud spec
 */
class Cloud {
	var $domain;
	var $port;
	var $path;
	var $registerProcedure;
	var $protocol;
	
	//getters -------
	function &getDomain() {
		return $this->domain;
	}
	
	function &getPort() {
		return $this->port;
	}
	
	function &getPath() {
		return $this->path;
	}
	
	function &getRegisterProcedure() {
		return $this->registerProcedure;
	}
	
	function &getProtocol() {
		return $this->protocol;
	}
}

/**
 * <image> (RSS)
 * 
 * An RSS image element is more complex than an Atom one. So in this one case,
 * the Atom images actually extend from this RSS object. It's inconsistent I
 * know. Sorry.
 * 
 * @link http://www.feedvalidator.org/docs/rss2.html#ltimagegtSubelementOfLtchannelgt RSS 2.0 image spec
 */
class Image {
	var $url;
	var $title;
	var $description;
	var $link;
	var $width = 88;
	var $height = 31;
	
	//getters -------
	function &get() {
		return $this->url;
	}
	
	function &getUrl() {
		return $this->url;
	}
	
	function &getTitle() {
		if (isset($this->title)) {
			return $this->title->get();
		} else {
			return null;
		}
	}
	
	function &getDescription() {
		if (isset($this->description)) {
			return $this->description->get();
		} else {
			return null;
		}
	}
	
	function &getLink() {
		if (isset($this->link)) {
			return $this->link[0]->getHref();
		} else {
			return null;
		}
	}
	
	function &getWidth() {
		return $this->width;
	}
	
	function &getHeight() {
		return $this->height;
	}
}

/**
 * <ttl> (RSS)
 * @link http://www.feedvalidator.org/docs/rss2.html#ltttlgtSubelementOfLtchannelgt RSS 2.0 ttl spec
 */
class Ttl {
	var $timeToLive;
	
	/**
	 * The content is always the time to live
	 * 
	 * @ignore
	 */
	function set_Content($_content) {
		$this->timeToLive = $_content;
	}
	
	//getters -------
	function &get() {
		return $this->timeToLive;
	}
}

/**
 * <textinput> (RSS)
 * @link http://www.feedvalidator.org/docs/rss2.html#lttextinputgtSubelementOfLtchannelgt RSS 2.0 textinput spec
 */
class TextInput {
	var $title;
	var $description;
	var $name;
	var $link;
	
	//getters -------
	function &getTitle() {
		if (isset($this->title)) {
			return $this->title->get();
		} else {
			return null;
		}
	}
	
	function &getDescription() {
		if (isset($this->description)) {
			return $this->description->get();
		} else {
			return null;
		}
	}
	
	function &getName() {
		return $this->name;
	}
	
	function &getLink() {
		if (isset($this->link)) {
			return $this->link[0]->getHref();
		} else {
			return null;
		}
	}
}

/**
 * <skiphours> (RSS)
 * @link http://backend.userland.com/skipHoursDays#skiphours Skip hours spec
 */
class SkipHours {
	var $hours;
	
	/**
	 * Rename to make clear this is an array
	 * 
	 * @ignore
	 */
	function setHour($hour) {
		$this->hours = $hour;
	}
	
	//getters ------
	function &get() {
		return $this->hours;
	}
	
	function &getHours() {
		return $this->hours;
	}
}

/**
 * <skipdays> (RSS)
 * @link http://backend.userland.com/skipHoursDays#skipdays Skip days spec
 */
class SkipDays {
	var $days;
	
	/**
	 * Rename to make clear this is an array
	 * 
	 * @ignore
	 */
	function setDay($day) {
		$this->days = $day;
	}
	
	//getters -------
	function &get() {
		return $this->days;
	}
	
	function &getDays() {
		return $this->days;
	}
}

/**#@-*/ // end RSS 2.0 optional


/**
 * <item> (RSS all versions)
 * 
 * Equivalent to atom:entry
 * 
 * @link http://www.feedvalidator.org/docs/rss2.html#hrelementsOfLtitemgt RSS
 * 2.0 spec
 */
class Item extends Entry {
	
	/**#@+
	 * Map RSS 2.0 elements to atom:entry
	 * 
	 * @ignore
	 */
	 
	function setGuid($guid) {
		$this->id = $guid;
		if ($guid->isPermaLink && ($this->getAltLink() == null)) {
			$altLink = new Seimour_Link();
			$altLink->href = $guid->_content;
			$altLink->rel = 'alternate';
			$altLink->type = 'text/html';
			$this->links[] = $altLink;
		}
	}
	
	function setDescription($description) {
		$this->summary = $description;
	}
	
	function setPubDate($pubDate) {
		$this->published = $pubDate;
	}
	
	function setIssued($issued) {
		$this->published = $issued;
	}
	
	function setCreated($created) {
		$this->created = $created;
	}
	
	function setEnclosure($enclosure) {
		$this->links[] = $enclosure;
	}
	
	/**#@-*/ // end RSS 2.0 mapping
	
	/**
	 * Map content encoded module to atom:entry
	 * 
	 * @ignore
	 */
	function setContent_Encoded($content_encoded) {
		$this->content = $content_encoded;
	}
	
	/**#@+
	 * Map dublin core module elements to atom:entry
	 * 
	 * @ignore
	 */
	 
	function setDc_title($dc_item) {
		$this->title = $dc_item;
	}
	
	function setDc_creator($dc_item) {
		$this->authors = $dc_item;
	}
	
	function setDc_subject($dc_item) {
		$this->categories = $dc_item;
	}
	
	function setDc_description($dc_item) {
		$this->summary = $dc_item;
	}
	
	function setDc_contributor($dc_item) {
		$this->contributors = $dc_item;
	}
	
	function setDc_date($dc_item) {
		$this->published = $dc_item;
	}
	
	function setDc_identifier($dc_item) {
		$this->id = $dc_item;
	}
	
	function setDc_source($dc_item) {
		$this->source = $dc_item;
	}
	
	function setDc_rights($dc_item) {
		$this->rights = $dc_item;
	}
	
	/**#@-*/ // end dublin core mapping
	
	function __wakeup() {
	}

}

/**
 * <enclosure> (RSS)
 * 
 * @link http://www.feedvalidator.org/docs/rss2.html#ltenclosuregtSubelementOfLtitemgt RSS 2.0 spec
 */
class Enclosure extends Link {
	
	/**
	 * Default atom:link properties
	 * @ignore
	 */
	function Enclosure() {
		$this->rel = 'enclosure';
	}
	
	/**
	 * Map to atom:link
	 * @ignore
	 */
	function setUrl($url) {
		$this->href = $url;
	}
}

/**
 * <guid> (RSS 2.0)
 * 
 * Equivalent to atom:id
 * 
 * @link http://www.feedvalidator.org/docs/rss2.html#ltguidgtSubelementOfLtitemgt RSS 2.0 spec
 */
class Guid extends Id {
	var $isPermaLink = true;
}


/**
 * <comments> (RSS)
 * 
 * @link http://www.feedvalidator.org/docs/rss2.html#comments RSS 2.0 spec
 */
class Comments extends CommonAttributes {
	var $url;
	
	/**
	 * Content is always a URL
	 * @ignore
	 */
	function set_Content($_content) {
		$this->url = $_content;
	}
	
	//getters -------
	function &get() {
		return $this->url;
	}
	
	function &getUrl() {
		return $this->url;
	}
}

// End RSS 2.0 exclusive objects ----------------------------------------------


// Begin RSS 1.0 + 0.9x exclusive objects -------------------------------------

/**
 * <rdf> (RSS 1.0)
 * 
 * Equivalent to atom:feed and RSS2 <rss>
 * 
 * Note: RSS 1 has a table of contents called 'items'. We discard it, since it
 * is kind of pointless.
 * 
 * @link http://web.resource.org/rss/1.0/spec#s5.2 RSS 1.0 spec
 */
class Rdf_RDF extends Rss {
	
	/**#@+
	 * Map RSS 1.0 elements to atom:feed
	 * @ignore
	 */
	 
	function setRdf_About($resource) {
		$selfLink = new Seimour_Link();
		$selfLink->href = $resource;
		$selfLink->rel = 'self';
		$selfLink->type = 'text/xml';
		$this->links[] = $selfLink;
	}
	
	function setItems($items) {
		// discard table of contents
	}
	
	function setItem($item) {
		$this->entries = $item;
	}
	
	function setImage($image) {
		$this->logo = $image;
	}
	
	/**#@-*/ // end RSS 1.0 mapping
	
}

// End RSS 1.0 + 0.9x exclusive objects -------------------------------------


// Begin RSS module objects -------------------------------------------------

/**
 * Content module element
 * @link http://web.resource.org/rss/1.0/modules/content/ Content module spec
 */
class Content_Encoded extends Content {
	function Content_Encoded() {
		$this->type = "html";
	}
}

/**#@+
 * Dublin core module element
 * @link http://web.resource.org/rss/1.0/modules/dc/ Dublin core module spec
 */
 
class dc_title extends Title {
}

class dc_creator extends Author {
	function set_Content($_content) {
		$this->name = $_content;
	}
}

class dc_subject extends Category {
	function set_Content($_content) {
		$this->term = $_content;
	}
}

class dc_description extends Description {
}

class dc_publisher {
}

class dc_contributor extends Contributor {
	function set_Content($_content) {
		$this->name = $_content;
	}
}

class dc_date extends Published {
}

class dc_type {
}

class dc_format {
}

class dc_identifier extends Id {
}

class dc_source extends Source {
}

//class dc_language {
//}

class dc_relation {
}

class dc_coverage {
}

class dc_rights extends Rights {
}

/**#@-*/ // end Dublin core module doc template

// End RSS module objects -------------------------------------------------



/**#@+
 * Supporting function
 */

/**
 * Return the href of the feed itself
 * 
 * @access	public
 * @param	array	$links	Array of hrefs associated with this feed
 * @return	string			URL of feed
 */
function findSelfHref($links) {
	$selfHref = null;
	foreach ($links as $link) {
		if ($link->rel == 'self') {
			$selfHref = $link->getHref();
			break;
		}
		if (empty($link->rel)) {
			$selfHref = $link->getHref();
		}
	}
	if ($selfHref == null && !empty($links)) {
		//assume the first link is the self link
		$selfHref = $this->links[0]->getHref();
	}
	return $selfHref;
}

/**
 * Return the alt href (which is usually the site publishing the feed)
 * 
 * @access	public
 * @param	array	$links	Array of hrefs associated with this feed/entry
 * @return	string			URL of feed/entry
 */
function findAltHref($links) {
	$altHref = null;
	foreach ($links as $link) {
		if ($link->rel == 'alternate') {
			$altHref = $link->getHref();
			break;
		}
		if (empty($link->rel)) {
			$altHref = $link->getHref();
		}
	}
	if ($altHref == null && !empty($links)) {
		//assume the first link is the alt link
		$altHref = $this->links[0]->getHref();
	}
	return $altHref;
}

/**
 * Strips dangerous tags from a HTML/XHTML string. This is to prevent XSS
 * attacks. Allowed tags include p, br, a, strong, b, em, i, u, ul, li, ol. All
 * others are stripped.
 * 
 * @access	public
 * @param	string	$html	HTML to process
 * @return	string			HTML with bad tags removed
 */
function removeBadTags($html) {
	$html = removeBadTagWithContent($html);
	$allowedTags='<p><br><br/><a><strong><b><em><i><u><ul><li><ol>';
	$html = strip_tags($html, $allowedTags);
	return preg_replace('/<(.*?)>/ie', "'<'.removeBadAttributes('\\1').'>'", $html);
}

/**
 * Strips script and style tags including the content within the tags.
 * 
 * @access	private
 * @param	string	$html	HTML to process
 * @return	string			HTML with bad tags and contents removed
 */
function removeBadTagWithContent($html) {
	return preg_replace('/(<script.*\/script>)|(<style.*\/style>)/i', '', $html);
}

/**
 * Strips dangerous attributes from tags which are otherwise safe. Basically, it
 * only allows the href in a tags.
 * 
 * @access	private
 * @param	string	$tagSource	Tag to process
 * @return	string				Tag with all bad attribs removed
 */
function removeBadAttributes($tagSource) {
	$stripAttrib = '/(\s\S+\s*=\s*("|\').*?(\2))/e'; 
	$tagSource = stripslashes($tagSource);
	$tagSource = preg_replace($stripAttrib, "removeAllButHref('\\1')", $tagSource);
	return $tagSource;
}

/**
 * Determines whether an attribute is safe. Which basically means it is an href
 * with no javascript:
 * 
 * @access	private
 * @param	string	$attrib	Attribute to test
 * @return	string			Sanitized attribute or blank string
 */
function removeAllButHref($attrib) {
	$attrib = stripslashes($attrib);
	if (strpos($attrib, 'href') === 1) {
		if (strpos($attrib, 'javascript:') === false) {
			return $attrib;
		}
	}
	return '';
}

/**
 * Cuts a large block of text down to the first 300 words. Adds an HTML ellipse
 * if necessary.
 * 
 * @access	private
 * @param	string	$text	Text to summarise
 * @return	string			Summarised text
 */
function summariseText($text) {
	if ( !empty($text) ) {			
		if (strlen($text) > 300) {
			$pat = "/(.|\n){1,300}\b/";
			preg_match( $pat, $text, $match );
			if (count($match) > 0) $text = trim($match[0]) . " &#8230;";
		}
	}
	return $text;
}

/**#@-*/ // end supporting functions doc template

?>